# SPDX-FileCopyrightText: 2023 Blender Authors
#
# SPDX-License-Identifier: GPL-2.0-or-later

from bpy.types import (
    Operator, # BFA
    Panel,
)

from bpy.props import EnumProperty # BFA


# BFA - Preset sizes for asset shelf thumbnails
thumbnail_sizes = {
    "TINY" : 32,      # Matches toolshelf
    "SMALL" : 64,     # Asset shelf default
    "MEDIUM" : 128,   # Asset browser default
    "LARGE" : 256,    # Maximum size
}

# BFA - Preset sizes and their corresponding UI labels
thumbnail_size_labels = tuple((item, item.title(), "") for item in thumbnail_sizes)


class ASSETSHELF_OT_change_thumbnail_size(Operator):
    bl_label = "Change Thumbnail Size"
    bl_idname = "asset_shelf.change_thumbnail_size"
    bl_description = "Change the size of thumbnails in discrete steps"

    size : EnumProperty(
        name="Thumbnail Size",
        description="Change the size of thumbnails in discrete steps",
        default="SMALL", # BFA - default to small
        items=thumbnail_size_labels
    )

    def execute(self, context):
        try:
            if not hasattr(context, 'asset_shelf'):
                self.report({'ERROR'}, "No asset shelf in current context")
                return {'CANCELLED'}

            shelf = context.asset_shelf
            if shelf is None:
                self.report({'ERROR'}, "No active asset shelf")
                return {'CANCELLED'}

            shelf.preview_size = thumbnail_sizes[self.size]
            return {'FINISHED'}

        except Exception as e:
            self.report({'ERROR'}, f"Error changing thumbnail size: {str(e)}")
            return {'CANCELLED'}

    @classmethod
    def poll(cls, context):
        return hasattr(context, 'asset_shelf') and context.asset_shelf is not None



class ASSETSHELF_PT_display(Panel):
    bl_label = "Display Settings"
    # Doesn't actually matter. Panel is instanced through popover only.
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'HEADER'

    def draw(self, context):
        layout = self.layout

        layout.use_property_split = True
        layout.use_property_decorate = False  # No animation.

        shelf = context.asset_shelf
        
        # BFA - Thumbnail preset buttons
        row = layout.row(align=True)
        row.context_pointer_set("asset_shelf", shelf)
        # BFA - operator_menu_enum doesn't seem to support horizontal button layout 
        # in popover panels, so the buttons are created one-by-one instead
        for value, label, _description in thumbnail_size_labels:
            row.operator("asset_shelf.change_thumbnail_size", text=label).size = value

        layout.prop(shelf, "preview_size", text="Size")
        layout.use_property_split = False
        layout.prop(shelf, "show_names", text="Names")

    @classmethod
    def poll(cls, context):
        return context.asset_shelf is not None


classes = (
    ASSETSHELF_PT_display,
    ASSETSHELF_OT_change_thumbnail_size # BFA
)


if __name__ == "__main__":  # only for live edit.
    from bpy.utils import register_class

    for cls in classes:
        register_class(cls)
