#!/usr/bin/env python3
"""
Test the final cleanup logic for the Bforartists Default Library addon.
This tests the find_and_cleanup_libraries logic without actually running Blender.
"""

import os
import sys
from os import path as p

def simulate_find_and_cleanup_libraries(existing_libraries, central_base):
    """
    Simulate the find_and_cleanup_libraries logic.
    
    Args:
        existing_libraries: list of dicts with keys 'name', 'path'
        central_base: string, the correct base path for libraries
    
    Returns:
        list of library names that need to be created
    """
    CENTRAL_LIB_SUBFOLDERS = [
        "Default Library",
        "Geometry Nodes Library", 
        "Shader Nodes Library",
        "Compositor Nodes Library"
    ]
    
    # Track which library names we need to create
    libraries_to_create = set(CENTRAL_LIB_SUBFOLDERS)
    
    # Track all libraries we find with our names
    found_libraries = []
    
    # First pass: Collect all libraries with our names (including .001, .002 suffixes)
    for i, lib in enumerate(existing_libraries):
        for lib_name in CENTRAL_LIB_SUBFOLDERS:
            # Check if library name matches (including .001, .002 suffixes)
            if lib['name'] == lib_name or lib['name'].startswith(f"{lib_name}."):
                found_libraries.append({
                    'index': i,
                    'original_index': i,  # Store original index for later reference
                    'name': lib['name'],
                    'clean_name': lib_name,  # The base name without .001 suffix
                    'path': lib['path'],
                    'correct_path': p.join(central_base, lib_name)
                })
                break
    
    # If no libraries found, we need to create all of them
    if not found_libraries:
        return list(libraries_to_create)
    
    # Group libraries by clean name (e.g., group "Default Library" and "Default Library.001")
    libraries_by_name = {}
    for lib in found_libraries:
        clean_name = lib['clean_name']
        if clean_name not in libraries_by_name:
            libraries_by_name[clean_name] = []
        libraries_by_name[clean_name].append(lib)
    
    # For each library name, determine which one(s) to keep and which to remove
    libraries_to_remove = []
    
    for clean_name, lib_list in libraries_by_name.items():
        # Sort libraries: first by if they have the correct path, then by name (prefer no suffix)
        lib_list.sort(key=lambda x: (
            x['path'] != x['correct_path'],  # Wrong path first
            x['name'] != clean_name,  # Suffixed names second
            x['name']  # Alphabetical as tiebreaker
        ))
        
        # Check if any library has the correct path
        correct_library = None
        for lib in lib_list:
            if lib['path'] == lib['correct_path']:
                correct_library = lib
                break
        
        if correct_library:
            # We have at least one library with the correct path
            # Remove all other libraries with this name (including .001, .002 suffixes)
            for lib in lib_list:
                if lib['index'] != correct_library['original_index']:
                    libraries_to_remove.append(lib['index'])
            
            # This library doesn't need to be created
            if clean_name in libraries_to_create:
                libraries_to_create.remove(clean_name)
        else:
            # No library has the correct path
            # Keep the first one (will be updated later) and remove all others
            if lib_list:
                # The first one after sorting will be updated
                library_to_update = lib_list[0]
                # Remove all others (if there are multiple)
                for lib in lib_list[1:]:
                    libraries_to_remove.append(lib['index'])
    
    # Now update any remaining libraries that don't have the correct path
    # We simulate this by checking the remaining libraries after removal
    # Note: In the real function, the indices change after removal, but we ignore that in simulation
    
    for i, lib in enumerate(existing_libraries):
        # Skip removed libraries
        if i in libraries_to_remove:
            continue
            
        for lib_name in CENTRAL_LIB_SUBFOLDERS:
            if lib['name'] == lib_name or lib['name'].startswith(f"{lib_name}."):
                correct_path = p.join(central_base, lib_name)
                if lib['path'] != correct_path:
                    # This library needs to be updated
                    # In the real function, it would be removed and re-added
                    # For simulation, we just mark that it doesn't need creation
                    if lib_name in libraries_to_create:
                        libraries_to_create.remove(lib_name)
                else:
                    # Library already has correct path
                    if lib_name in libraries_to_create:
                        libraries_to_create.remove(lib_name)
                break
    
    return list(libraries_to_create)


def test_scenario(name, existing_libraries, central_base, expected_creations):
    """Test a single scenario."""
    print(f"\n{name}")
    print("-" * 40)
    
    result = simulate_find_and_cleanup_libraries(existing_libraries, central_base)
    
    print(f"Existing libraries:")
    for lib in existing_libraries:
        print(f"  - {lib['name']}: {lib['path']}")
    
    print(f"\nCentral base: {central_base}")
    print(f"Libraries to create: {result}")
    print(f"Expected to create: {expected_creations}")
    
    # Sort both lists for comparison
    result_sorted = sorted(result)
    expected_sorted = sorted(expected_creations)
    
    if result_sorted == expected_sorted:
        print("✅ Test passed!")
        return True
    else:
        print(f"❌ Test failed! Got {result_sorted}, expected {expected_sorted}")
        return False


def run_all_tests():
    """Run all test scenarios."""
    print("Testing find_and_cleanup_libraries logic")
    print("=" * 60)
    
    all_passed = True
    
    # Scenario 1: Fresh install - no libraries exist
    all_passed &= test_scenario(
        "Scenario 1: Fresh install",
        [],
        "/user/prefs/asset_libraries/bfa_central_asset_library",
        ["Default Library", "Geometry Nodes Library", "Shader Nodes Library", "Compositor Nodes Library"]
    )
    
    # Scenario 2: All libraries exist with correct paths
    all_passed &= test_scenario(
        "Scenario 2: All libraries correct",
        [
            {"name": "Default Library", "path": "/user/prefs/asset_libraries/bfa_central_asset_library/Default Library"},
            {"name": "Geometry Nodes Library", "path": "/user/prefs/asset_libraries/bfa_central_asset_library/Geometry Nodes Library"},
            {"name": "Shader Nodes Library", "path": "/user/prefs/asset_libraries/bfa_central_asset_library/Shader Nodes Library"},
            {"name": "Compositor Nodes Library", "path": "/user/prefs/asset_libraries/bfa_central_asset_library/Compositor Nodes Library"}
        ],
        "/user/prefs/asset_libraries/bfa_central_asset_library",
        []
    )
    
    # Scenario 3: Some libraries have wrong paths
    all_passed &= test_scenario(
        "Scenario 3: Some wrong paths",
        [
            {"name": "Default Library", "path": "/old/path/Default Library"},
            {"name": "Geometry Nodes Library", "path": "/user/prefs/asset_libraries/bfa_central_asset_library/Geometry Nodes Library"},
            {"name": "Shader Nodes Library", "path": "/addon/folder/Shader Nodes Library"},
            # Compositor Nodes Library doesn't exist
        ],
        "/user/prefs/asset_libraries/bfa_central_asset_library",
        ["Compositor Nodes Library"]  # Only missing one
    )
    
    # Scenario 4: Duplicate libraries with .001 suffixes
    all_passed &= test_scenario(
        "Scenario 4: Duplicates with suffixes",
        [
            {"name": "Default Library", "path": "/user/prefs/asset_libraries/bfa_central_asset_library/Default Library"},
            {"name": "Default Library.001", "path": "/old/path/Default Library"},
            {"name": "Geometry Nodes Library.001", "path": "/some/path/Geometry Nodes Library"},
            {"name": "Geometry Nodes Library.002", "path": "/another/path/Geometry Nodes Library"},
            {"name": "Shader Nodes Library", "path": "/user/prefs/asset_libraries/bfa_central_asset_library/Shader Nodes Library"},
        ],
        "/user/prefs/asset_libraries/bfa_central_asset_library",
        ["Compositor Nodes Library"]  # Only missing one
    )
    
    # Scenario 5: All libraries have wrong paths (migration scenario)
    all_passed &= test_scenario(
        "Scenario 5: All wrong paths (migration)",
        [
            {"name": "Default Library", "path": "/old/system/path/Default Library"},
            {"name": "Geometry Nodes Library", "path": "/old/system/path/Geometry Nodes Library"},
            {"name": "Shader Nodes Library", "path": "/old/system/path/Shader Nodes Library"},
            {"name": "Compositor Nodes Library", "path": "/old/system/path/Compositor Nodes Library"}
        ],
        "/user/prefs/asset_libraries/bfa_central_asset_library",
        []  # All will be updated, none need creation
    )
    
    # Scenario 6: Mixed with some correct, some wrong, some duplicates
    all_passed &= test_scenario(
        "Scenario 6: Complex mixed scenario",
        [
            {"name": "Default Library", "path": "/user/prefs/asset_libraries/bfa_central_asset_library/Default Library"},
            {"name": "Default Library.001", "path": "/old/path/Default Library"},
            {"name": "Geometry Nodes Library", "path": "/wrong/path/Geometry Nodes Library"},
            {"name": "Shader Nodes Library.001", "path": "/user/prefs/asset_libraries/bfa_central_asset_library/Shader Nodes Library"},
            {"name": "Shader Nodes Library.002", "path": "/very/old/path/Shader Nodes Library"},
            # Compositor Nodes Library doesn't exist
        ],
        "/user/prefs/asset_libraries/bfa_central_asset_library",
        ["Compositor Nodes Library"]  # Only missing one
    )
    
    print("\n" + "=" * 60)
    if all_passed:
        print("✅ All tests passed!")
    else:
        print("❌ Some tests failed!")
    
    return all_passed


if __name__ == "__main__":
    success = run_all_tests()
    sys.exit(0 if success else 1)